/* Copyright (C) 2014-2018 RealVNC Ltd.  All Rights Reserved.
 */

#ifndef UUID_a8f7d1c8_0129_11e4_8b37_f7f7140abe98
#define UUID_a8f7d1c8_0129_11e4_8b37_f7f7140abe98

/**
 * \file vncwifidisplaysdk.h
 *
 * This is the main header file for the VNC WiFi Display SDK. To use the SDK,
 * you need only include this file.
 *
 * \see VNCWiFiDisplaySDKInitialize, VNCWiFiDisplaySDK
 */

/**
 * \mainpage VNC WiFi Display SDK - Native Interface
 *
 * \section section_contents Contents
 *
 *  -# \ref section_overview
 *  -# \ref section_usage
 *  -# \ref section_thread
 *  -# \ref section_decoder_plugins
 *  -# \ref section_codecs
 *  -# \ref section_wfd_source
 *  -# \ref section_wfd_session
 *  -# \ref section_uibc_mirrorlink
 *  -# \ref section_legal
 *
 * \section section_overview Overview
 *
 * The VNC WiFi Display SDK allows you to create and manipulate VNC WiFi
 * Display Sink objects. A VNC WiFi Display Sink connects to a WiFi Display
 * source over a network, establishing a WiFi Display session. During this
 * session, the sink consumes, decodes and renders the RTP media stream that
 * the source produces. Where the WiFi Display source is a MirrorLink device,
 * the VNC WiFi Display Sink may also send and receive events across a User
 * Input Back Channel (UIBC). This can be used to facilitate remote user
 * control of the WiFi Display source. This SDK does not currently support UIBC
 * using non-MirrorLink input categories.
 *
 * The RTP media stream provided by the WiFi Display source consists of an
 * MPEG2 Transport Stream (MPEG2-TS) which contains within it a single audio
 * and/or video elementary stream (support for a video stream is mandatory for
 * primary sinks). Decoding of the MPEG2-TS is performed by a VNC WiFi Display
 * Decoder plugin, which must be registered with a VNC WiFi Display Sink before
 * it is started.
 *
 * This SDK does not perform discovery of WiFi Display sources. You must obtain
 * details of the WiFi Display source to use separately, and pass the details
 * of the source to the SDK. You may wish to use the VNC WiFi P2P Discoverer to
 * discover available WiFi Display sources.
 *
 * \section section_usage Basic usage
 *
 * \subsection subsection_usage_compiling Compiling your application
 *
 * To use the SDK, your application must first define exactly one of the
 * preprocessor macros VNC_USE_STDINT_H or VNC_USE_DEFAULT_INT_TYPEDEFS. These
 * macros govern the definitions of the vnc_*_t integer types defined by
 * vncint.h.
 *
 *  - If your build environment provides the ISO C99 header stdint.h, or you
 *    can author an equivalent yourself, you should define VNC_USE_STDINT_H.
 *    This is the recommended approach for defining the SDK's integer types.
 *  - If you examine vncint.h and find that the types used when you define
 *    VNC_USE_DEFAULT_INT_TYPEDEFS are correct for your platform, then you may
 *    wish to define VNC_USE_DEFAULT_INT_TYPEDEFS instead of VNC_USE_STDINT_H.
 *    Additionally, if you are targeting a 64-bit platform with a compiler
 *    other than Microsoft Visual C++ or gcc, then you may have to define
 *    VNC_FORCE_64_BIT in order to define vnc_intptr_t and vnc_uintptr_t
 *    correctly.  See vncint.h itself for further information.
 *
 * The types defined by VNC_USE_DEFAULT_INT_TYPEDEFS are known to be correct
 * for Microsoft Visual C++ 2005.
 *
 * After defining either VNC_USE_STDINT_H or VNC_USE_DEFAULT_INT_TYPEDEFS, your
 * application should include vncwifidisplaysdk.h. You are encouraged to use
 * the VNCVerifyIntegerTypes() macro to verify that the integer types have been
 * defined correctly.
 *
 * \subsection subsection_usage_startup Initializing the SDK
 *
 * -# Load the SDK DLL (necessary only if you did not link against it).
 *    - on Linux, call dlopen()
 * -# Locate the SDK entry point, VNCWiFiDisplaySDKInitialize().
 *    - on Linux, call dlsym()
 * -# Call VNCWiFiDisplaySDKInitialize(). This populates a VNCWiFiDisplaySDK
 *    structure with pointers to the rest of the SDK's 'methods'.
 *
 * Typically, your application will call VNCWiFiDisplaySDKInitialize() at
 * start-up. You should not call VNCWiFiDisplaySDKInitialize() more than once.
 *
 * \subsection subsection_usage_creating Creating a VNC WiFi Display Sink object.
 *
 * Call VNCWiFiDisplaySinkCreate() to create a VNC WiFi Display Sink object.
 * You may reuse the same VNC WiFi Display Sink object for a succession of
 * sessions that require a similar configuration. There is no need to create a
 * new VNC WiFi Display Sink for each session.
 *
 * If you need to manage more than one WiFi Display sink in parallel, you can
 * call VNCWiFiDisplaySinkCreate() as often as needed to create the required
 * number of VNC WiFi Display Sink instances. However, note that you should not
 * call VNCWiFiDisplaySDKInitialize() more than once.
 *
 * When you create a VNC WiFi Display Sink object, you supply pointers to
 * various callbacks that provide your application with information about
 * particular events. All callbacks are optional, however it is highly
 * recommended that you implement at least the following callbacks:
 *
 * - VNCWiFiDisplaySinkErrorCallback(), so that you are notified when the WiFi
 *   Display sink thread stops.
 * - VNCWiFiDisplaySinkStateChangedCallback(), so that you are notified when
 *   the state of the VNC WiFi Display Sink changes, for example because the
 *   RTP media stream has been paused or resumed.
 *
 * If you wish to support interaction with a MirrorLink WiFi Display source,
 * then you must also implement the following callbacks:
 *
 * - VNCWiFiDisplaySinkUIBCMirrorLinkConfigurationSelectedCallback(), so that
 *   you are notified of the MirrorLink UIBC configuration that the MirrorLink
 *   WiFi Display source has selected.
 * - VNCWiFiDisplaySinkUIBCStateChangedCallback(), so that you are notified
 *   when the User Input Back Channel with the WiFi Display source is active.
 * - VNCWiFiDisplaySinkMirrorLinkUIContextEventCallback(), so that you are
 *   notified when the MirrorLink WiFi Display source sends User Interface
 *   Context events, and switch to a native user interface if required to by
 *   such an event.
 *
 * After creating a VNC WiFi Display Sink object, you may find it useful to use
 * VNCWiFiDisplaySinkSetContext() to associate some application-specific data
 * with the sink. For example, you could pass a pointer to a struct that holds
 * state for your application (window handles, log file handle, etc). It is
 * recommended that any call to VNCWiFiDisplaySinkSetContext() is made
 * immediately after VNCWiFiDisplaySinkCreate(), so that the context pointer is
 * available to all callbacks that the SDK may make.
 *
 * The error codes that an SDK API may return are described in the API 
 * documentation. However, any SDK API may also return the error
 * VNCWiFiDisplayErrorFailed if it encounters an unexpected and fatal error.
 * One reason that an API may return VNCWiFiDisplayErrorFailed is that an
 * out of memory condition occurred, which prevented the SDK from fulfilling
 * the request.
 *
 * \subsection subsection_usage_licensing Licensing the SDK
 *
 * In order to use the SDK, your application must provide each VNC WiFi Display
 * Sink instance with at least one license. Use VNCWiFiDisplaySinkAddLicense()
 * to do this. Contact your RealVNC sales representative to obtain a license.
 *
 * \subsection subsection_usage_configuring Configuring a WiFi Display sink
 *
 * Use VNCWiFiDisplaySinkSetParameter() to set any desired parameters on the
 * VNC WiFi Display Sink. See vncwifidisplayparameters.h for a full list of
 * configuration options.
 *
 * Use VNCWiFiDisplaySinkSetNativeResolution() and
 * VNCWiFiDisplaySinkSetActiveConnectorType() to specify the details of the
 * display to which you will render.
 *
 * Use VNCWiFiDisplaySinkRegisterDecoder() or
 * VNCWiFiDisplaySinkLoadDecoderFromFile() to register a single VNC WiFi
 * Display Decoder Plugin, which supports the decoding and rendering of audio
 * and H.264 codecs. See \ref section_decoder_plugins for further information.
 *
 * Use VNCWiFiDisplaySinkAddAudioCodec() and VNCWiFiDisplaySinkAddH264Codec()
 * to specify the audio and H.264 codecs that you wish to support. Each codec
 * must be supported by the decoder plugin that you have registered. See
 * \ref section_codecs for further information.
 *
 * Use VNCWiFiDisplaySinkSetSource() to specify which WiFi Display source to
 * connect to. You must obtain this information separately, as this SDK does
 * not support discovery of WiFi Display sources. You may wish to use the VNC
 * WiFi P2P Discoverer to discover available WiFi Display sources.
 *
 * \section section_thread The WiFi Display sink thread.
 *
 * Calling VNCWiFiDisplaySinkStart() starts a new thread of execution. This
 * thread is called the WiFi Display sink thread. This thread shall establish
 * and maintain a WiFi Display session with the WiFi Display source.
 *
 * Once started, the WiFi Display sink thread continues to execute until one of
 * the following occurs:
 *
 * - Your application calls VNCWiFiDisplaySinkStop() (including implicit calls
 *   to VNCWiFiDisplaySinkStop() made by VNCWiFiDisplaySinkDestroy().
 * - An error occurs, which prevents the VNC WiFi Display Sink from continuing.
 * - The WiFi Display source tears down the WiFi Display session.
 *
 * Before it exits, the WiFi Display sink thread will always make exactly one
 * call to your application's \ref VNCWiFiDisplaySinkErrorCallback. This call
 * will take place even if your application calls VNCWiFiDisplaySinkStop() or
 * VNCWiFiDisplaySinkDestroy().
 *
 * \subsection subsection_thread_app Interaction with your application
 *
 * All SDK processing for each WiFi Display sink instance takes place within
 * the WiFi Display sink thread for that instance. The majority of WiFi Display
 * sink APIs place a command on a queue that the WiFi Display sink thread
 * services, and do not return to the caller until the command has been
 * serviced by the WiFi Display sink thread.
 *
 * The only exception to this rule is the VNCWiFiDisplaySinkLogCallback(),
 * which the SDK may make from any thread, including calls made before the WiFi
 * Display sink thread is started. However, the SDK guarantees that no two
 * threads will enter a VNCWiFiDisplaySinkLogCallback() at the same time, even
 * if there are multiple sink instances.
 *
 * This threading model greatly simplifies the interaction of the SDK with your
 * application.  It does, however, mean that there are three particular rules
 * that your application must follow in order to avoid deadlock.
 *
 * \subsection subsection_thread_rule1 Don't call SDK API while holding locks
 *
 * If an application thread calls an SDK API whilst holding a lock that is
 * needed by a callback function, then the likelihood of deadlock is very high.
 * The WiFi Display sink thread may invoke that callback function, causing it
 * to block until it acquires the lock. This prevents the WiFi Display sink
 * thread from servicing the application thread's API calls, leading to
 * deadlock.
 *
 * \subsection subsection_thread_rule2 Don't block the WiFi Display sink thread to wait for a response from an application thread
 *
 * It is common to want to communicate events from the WiFi Display sink thread
 * to the application thread. If the inter-thread RPC mechanism used to
 * communicate the events to the application thread blocks until it is
 * serviced, then the likelihood of deadlock is very high. Sooner or later, the
 * application thread will be blocked on an SDK API call at the time that the
 * WiFi Display sink thread needs it to service the blocking RPC, and the
 * application is deadlocked.
 *
 * \subsection subsection_thread_rule3 Don't attempt to destroy a VNC WiFi Display Sink from an SDK callback
 *
 * VNCWiFiDisplaySinkDestroy() shall block until the WiFi Display sink thread
 * has terminated. If you call this API from an SDK callback, then the sink
 * thread shall wait for itself to terminate, resulting in deadlock.
 *
 * If you wish to destroy a VNC WiFi Display Sink instance in response to an
 * SDK callback, then your callback implementation should signal your
 * application thread to invoke VNCWiFiDisplaySinkDestroy().
 *
 * \section section_decoder_plugins Decoder plugins
 *
 * A VNC WiFi Display Decoder Plugin must be registered with a VNC WiFi Display
 * sink, to decode and render the incoming MPEG2-TS from the WiFi Display
 * source. The decoder plugin shall declare to the SDK the audio and H.264
 * codecs that it supports, and you may choose to advertise any or all of these
 * codecs to the WiFi Display source during the WiFi Display capability
 * negotiation phase.
 *
 * You must call either VNCWiFiDisplaySinkRegisterDecoder() or
 * VNCWiFiDisplaySinkLoadDecoderFromFile() to register the decoder plugin. Each
 * plugin registered must be of a unique type. Once the decoder plugin has been
 * registered, you can use VNCWiFiDisplaySinkQueryDecoderAudioCodecs() and
 * VNCWiFiDisplaySinkQueryDecoderH264Codecs() to query the audio and H.264
 * codecs that the plugin supports. You may then decide to add any or all of
 * these codecs to the VNC WiFi Display Sink using
 * VNCWiFiDisplaySinkAddAudioCodec() and VNCWiFiDisplaySinkAddH264Codec().
 * Only the codecs that have been added to the sink in this way will be
 * advertised to WiFi Display sources.
 *
 * \section section_codecs Codecs
 *
 * In order to present audio and H.264 codecs to the WiFi Display source, you
 * must first add them to the VNC WiFi Display Sink by calling
 * VNCWiFiDisplaySinkAddAudioCodec() or VNCWiFiDisplaySinkAddH264Codec(). Audio
 * codecs are optional, and should only be added if you wish to implement an
 * audio-capable sink. However, at least one H.264 codec must be added before a
 * VNC WiFi Display Sink can be started.
 *
 * The SDK shall ensure that each audio and H.264 codec that you add is
 * supported by the decoder plugin that you have registered. You must therefore 
 * register the decoder plugin that you wish to use with the VNC WiFi Display
 * Sink first, before attempting to add codecs.
 *
 * \section section_wfd_source WiFi Display source
 *
 * You must specify which WiFi Display source to connect to before you can
 * start the VNC WiFi Display Sink. This SDK does not perform discovery of WiFi
 * Display sources. You may wish to use the VNC WiFi P2P Discoverer for this
 * purpose. Once you have identified the WiFi Display source that you wish to
 * use, call VNCWiFiDisplaySinkSetSource() to specify its IP address, RTSP
 * control port, and the local IP address that the VNC WiFi Display Sink should
 * use when communicating with the source. The VNC WiFi Display Sink will not
 * attempt to connect to the source until the WiFi Display sink thread is
 * started with VNCWiFiDisplaySinkStart().
 *
 * \section section_wfd_session WiFi Display session
 *
 * Once the WiFi Display sink thread is started, it will attempt to establish a
 * WiFi Display session with the WiFi Display source. The SDK shall notify your
 * application when the session is established by invoking the
 * VNCWiFiDisplaySinkStateChangedCallback() with the state parameter set to
 * VNCWiFiDisplayStatePlaying. The WiFi Display source shall then start the RTP
 * media stream.
 *
 * Whilst the WiFi Display session is active, you may pause the RTP media
 * stream at any time by calling VNCWiFiDisplaySinkPause(). The WiFi Display
 * source may also request that the media stream be paused at any time. When
 * the RTP media stream is paused, the SDK shall notify your application by
 * invoking the VNCWiFiDisplaySinkStateChangedCallback() with the state
 * parameter set to VNCWiFiDisplayStatePaused. A paused RTP media stream may be
 * resumed by calling VNCWiFiDisplaySinkResume().
 *
 * A WiFi Display session may be torn down by the WiFi Display source, or by
 * your application calling VNCWiFiDisplaySinkStop() or
 * VNCWiFiDisplaySinkDestroy(). When this occurs, or a fatal error occurs, then
 * the WiFi Display sink thread shall stop. The SDK shall invoke the
 * VNCWiFiDisplaySinkErrorCallback() to notify your application.
 *
 * \section section_uibc_mirrorlink UIBC MirrorLink connections
 *
 * If the WiFi Display source is a MirrorLink device, then a VNC WiFi Display
 * Sink may establish a UIBC connection with the source for MirrorLink input.
 * A UIBC connection can be used to send MirrorLink key, pointer or touch events
 * to remotely control a user interface on the source, as well as status, blocking
 * and cut text event notifications. The VNC WiFi Display sink may also receive
 * status, context, cut text and text output event notifications from the
 * source. Note that if you are implementing a MirrorLink client, then UIBC
 * support is mandatory. In addition, the text output event notification is
 * deprecated in MirrorLink 1.3, SDK users should not expect the text
 * output notification in MirrorLink 1.3 or later sessions.
 *
 * To enable UIBC MirrorLink support on a VNC WiFi Display Sink, you must call
 * VNCWiFiDisplaySinkSetUIBCMirrorLinkConfiguration() with an appropriate
 * configuration for your platform. If the WiFi Display source supports UIBC
 * MirrorLink connections, then it shall select a MirrorLink configuration that is
 * compatible with that specified in the call to
 * VNCWiFiDisplaySinkSetUIBCMirrorLinkConfiguration(). The SDK shall notify
 * your application when this occurs by calling the
 * VNCWiFiDisplaySinkUIBCMirrorLinkConfigurationSelectedCallback(). Once the
 * UIBC connection has been established, the SDK shall call the
 * VNCWiFiDisplaySinkUIBCStateChangedCallback() with
 * VNCWiFiDisplaySinkUIBCStateActive. After receiving this callback,
 * your application may use SDK APIs, and received callbacks, that require a
 * UIBC connection.
 *
 * Remote control of a MirrorLink WiFi Display source's user interface may be
 * accomplished with VNCWiFiDisplaySinkSendMirrorLinkKeyEvent(),
 * VNCWiFiDisplaySinkSendMirrorLinkPointerEvent() and
 * VNCWiFiDisplaySinkSendMirrorLinkTouchEvent(). Blocking events can be
 * generated with VNCWiFiDisplaySinkSendMirrorLinkUIBlockingEvent() and
 * VNCWiFiDisplaySinkSendMirrorLinkAudioBlockingEvent(). You may also use
 * VNCWiFiDisplaySinkSendMirrorLinkSinkStatusEvent() to update the source on
 * the current device status, and
 * VNCWiFiDisplaySinkSendMirrorLinkSinkCutTextEvent() to place text in the
 * source's host clipboard.
 *
 * \section section_legal Legal information
 *
 * Copyright (C) 2014-2018 RealVNC Ltd. All Rights Reserved.
 *
 * RealVNC and VNC are trademarks of RealVNC Limited and are protected by
 * trademark registrations and/or pending trademark applications in the
 * European Union, United States of America and other jurisdictions.
 *
 * MirrorLink is a registered trademark of Car Connectivity Consortium in the
 * U.S. and in other countries.
 *
 * For more information on WiFi Display, please refer to the "Wi-Fi Display
 * Technical Specification" published by "Wi-Fi Alliance" at
 * http://www.wi-fi.org.
 *
 * Other trademarks are the property of their respective owners.
 *
 * Protected by UK patents 2481870, 2491657; US patents 8760366, 9137657; EU patent 2652951.
 *
 * \see VNCWiFiDisplaySDK, VNCWiFiDisplaySinkCallbacks,
 * VNCWiFiDisplaySDKInitialize, VNCWiFiDisplaySDKUninitialize,
 * VNCVerifyIntegerTypes
 */

#include "vncwifidisplaysink.h"
#include "vncwifidisplaytypes.h"

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * \brief Uninitializes the SDK.
 *
 * You must call this function after destroying all VNC WiFi Display Sinks but
 * before unloading the SDK.
 */
typedef void VNCCALL
VNCWiFiDisplaySDKUninitialize(void);

/**
 * \brief Returns a symbolic name corresponding to an SDK error code.
 *
 * For example, if the error is VNCWiFiDisplayErrorInvalidParameter, then the
 * returned string is "InvalidParameter". Such strings may be useful as
 * internationalization keys or in log entries.
 *
 * The strings returned by this method are brief and are not internationalized.
 * They are therefore not suitable for display to an end user.
 *
 * \param error The error code.
 *
 * \return A short string corresponding to the error code. The memory pointed
 * to is owned by the SDK; do not free it. If the error code is not recognized
 * by the SDK, then the return value is NULL.
 */
typedef const char *VNCCALL
VNCWiFiDisplayGetErrorName(VNCWiFiDisplayError error);

/**
 * \brief Allocates a string for use by the SDK.
 *
 * Use this method to allocate strings whose ownership must be passed to the
 * SDK.
 *
 * \param string A string to copy. See the note below.
 * \param length The length of the string to allocate. See the note below.
 *
 * \return Returns a new string. The returned string must be freed by calling
 * VNCWiFiDisplayFreeString().
 *
 * \note There are three ways to use this function:
 *
 *  - string != NULL, length == (size_t) -1 - copies an existing NUL-terminated
 *    string.
 *  - string != NULL, length >= 0 - copies an existing string of that length
 *    and places a NUL-terminator afterwards.
 *  - string == NULL, length >= 0 - allocates a buffer that can hold a string
 *    of that length and also a NUL-terminator (i.e. it allocates (length + 1)
 *    bytes).  The entire buffer is initialized with NULs.
 */
typedef char *VNCCALL
VNCWiFiDisplayAllocString(const char *string, size_t length);

/**
 * \brief Frees a string that has been allocated by the SDK or by calling
 * VNCWiFiDisplayAllocString().
 *
 * Use this method to free any strings that have been allocated in this way but
 * are no longer required.
 *
 * It is guaranteed to be safe to call VNCWiFiDisplayFreeString() with a
 * NULL parameter.
 *
 * \param string The string to free.
 *
 * \note Unlike other VNC Mobile Solution SDKs, the implementation of this API
 * does *not* zero the memory prior to freeing it.
 */
typedef void VNCCALL
VNCWiFiDisplayFreeString(char *string);

/**
 * \brief Contains the SDK's version and addresses of the SDK's methods.
 *
 * An instance of this structure should be passed to
 * VNCWiFiDisplaySDKInitialize(). A successful call to
 * VNCWiFiDisplaySDKInitialize() fills in the structure with the SDK's major
 * and minor version numbers and the addresses of the SDK's methods.
 *
 * After the structure is initialized, you may create a VNC WiFi Display Sink
 * and interact with it by calling the SDK's methods through the function
 * pointers in the structure.  Begin by calling VNCWiFiDisplaySinkCreate().
 *
 * You may also wish to check the version numbers reported by the SDK.  These
 * numbers are the version of the SDK itself, and are not related to WiFi
 * Display or MirrorLink version numbers.
 *
 * For full documentation on each of the function pointers in the
 * VNCWiFiDisplaySDK structure, see the documentation for the corresponding
 * typedef.
 *
 * \see VNCWiFiDisplaySDKInitialize, VNCWiFiDisplaySDKUninitialize
 * \see VNCWiFiDisplaySinkCreate
 */
typedef struct
{
  /**
   * \brief Major component of the SDK's version number.
   */
  vnc_int32_t versionMajor;

  /**
   * \brief Minor component of the SDK's version number.
   */
  vnc_int32_t versionMinor;

  /**
   * \brief Patch component of the SDK's version number.
   */
  vnc_int32_t versionPatch;

  /**
   * \brief Build number component of the SDK's version number.
   */
  vnc_int32_t versionBuild;

  /**
   * \brief Uninitializes the SDK.
   */
  VNCWiFiDisplaySDKUninitialize *vncWiFiDisplaySDKUninitialize;

  /**
   * \brief Returns a symbolic name corresponding to an SDK error code.
   */
  VNCWiFiDisplayGetErrorName *vncWiFiDisplayGetErrorName;

  /**
   * \brief Allocates a string for use by the SDK.
   */
  VNCWiFiDisplayAllocString *vncWiFiDisplayAllocString;

  /**
   * \brief Frees a string that has been allocated by the SDK or by calling
   * VNCWiFiDisplayAllocString().
   */
  VNCWiFiDisplayFreeString *vncWiFiDisplayFreeString;

  /**
   * \brief Create a new VNC WiFi Display Sink instance.
   */
  VNCWiFiDisplaySinkCreate *vncWiFiDisplaySinkCreate;

  /**
   * \brief Destroy an instance of the VNC WiFi Display Sink.
   */
  VNCWiFiDisplaySinkDestroy *vncWiFiDisplaySinkDestroy;

  /**
   * \brief Associates a context pointer with a VNC WiFi Display Sink.
   */
  VNCWiFiDisplaySinkSetContext *vncWiFiDisplaySinkSetContext;

  /**
   * \brief Retrieves the VNC WiFi Display Sink's context pointer.
   */
  VNCWiFiDisplaySinkGetContext *vncWiFiDisplaySinkGetContext;

  /**
   * \brief Adds a VNC license to a VNC WiFi Display Sink.
   */
  VNCWiFiDisplaySinkAddLicense *vncWiFiDisplaySinkAddLicense;

  /**
   * \brief Queries a VNC WiFi Display Sink for the value of an integer-valued
   * runtime property.
   */
  VNCWiFiDisplaySinkGetProperty *vncWiFiDisplaySinkGetProperty;

  /**
   * \brief Queries a VNC WiFi Display Sink for the value of a string-valued
   * runtime property.
   */
  VNCWiFiDisplaySinkGetPropertyString *vncWiFiDisplaySinkGetPropertyString;

  /**
   * \brief Sets the value of a VNC WiFi Display Sink parameter.
   */
  VNCWiFiDisplaySinkSetParameter *vncWiFiDisplaySinkSetParameter;

  /**
   * \brief Queries the value of a VNC WiFi Display Sink parameter.
   */
  VNCWiFiDisplaySinkGetParameter *vncWiFiDisplaySinkGetParameter;

  /**
   * \brief Sets the native resolution for a VNC WiFi Display Sink.
   */
  VNCWiFiDisplaySinkSetNativeResolution *vncWiFiDisplaySinkSetNativeResolution;

  /**
   * \brief Sets the active connector type for a VNC WiFi Display Sink.
   */
  VNCWiFiDisplaySinkSetActiveConnectorType *vncWiFiDisplaySinkSetActiveConnectorType;

  /**
   * \brief Registers a VNC WiFi Display Decoder Plugin for a VNC WiFi Display
   * Sink.
   */
  VNCWiFiDisplaySinkRegisterDecoder *vncWiFiDisplaySinkRegisterDecoder;

  /**
   * \brief Registers a VNC WiFi Display Decoder Plugin for a VNC WiFi Display
   * Sink.
   */
  VNCWiFiDisplaySinkLoadDecoderFromFile *vncWiFiDisplaySinkLoadDecoderFromFile;
  
  /**
   * \brief Sets a property of a WiFi Display decoder.
   */
  VNCWiFiDisplaySinkSetDecoderProperty *vncWiFiDisplaySinkSetDecoderProperty;

  /**
   * \brief Queries the registered VNC WiFi Display Decoder Plugin for the set
   * of audio codecs that it supports.
   */
  VNCWiFiDisplaySinkQueryDecoderAudioCodecs *vncWiFiDisplaySinkQueryDecoderAudioCodecs;

  /**
   * \brief Queries the registered VNC WiFi Display Decoder Plugin for the set
   * of H.264 codecs that it supports.
   */
  VNCWiFiDisplaySinkQueryDecoderH264Codecs *vncWiFiDisplaySinkQueryDecoderH264Codecs;

  /**
   * \brief Adds an audio codec to a VNC WiFi Display Sink.
   */
  VNCWiFiDisplaySinkAddAudioCodec *vncWiFiDisplaySinkAddAudioCodec;

  /**
   * \brief Clears all audio codecs currently associated with a VNC WiFi
   * Display Sink.
   */
  VNCWiFiDisplaySinkClearAudioCodecs *vncWiFiDisplaySinkClearAudioCodecs;

  /**
   * \brief Adds a H.264 video codec to a VNC WiFi Display Sink.
   */
  VNCWiFiDisplaySinkAddH264Codec *vncWiFiDisplaySinkAddH264Codec;

  /**
   * \brief Clears all H.264 codecs currently associated with a VNC WiFi
   * Display Sink.
   */
  VNCWiFiDisplaySinkClearH264Codecs *vncWiFiDisplaySinkClearH264Codecs;

  /**
   * \brief Sets the MirrorLink configuration that a VNC WiFi Display Sink
   * should use for User Input Back Channel (UIBC) sessions.
   */
  VNCWiFiDisplaySinkSetUIBCMirrorLinkConfiguration *vncWiFiDisplaySinkSetUIBCMirrorLinkConfiguration;

  /**
   * \brief Sets the WiFi Display source that a VNC WiFi Display Sink should
   * connect to.
   */
  VNCWiFiDisplaySinkSetSource *vncWiFiDisplaySinkSetSource;

  /**
   * \brief Starts a VNC WiFi Display Sink.
   */
  VNCWiFiDisplaySinkStart *vncWiFiDisplaySinkStart;

  /**
   * \brief Stops a VNC WiFi Display Sink.
   */
  VNCWiFiDisplaySinkStop *vncWiFiDisplaySinkStop;

  /**
   * \brief Pauses the RTP media stream received by a VNC WiFi Display Sink.
   */
  VNCWiFiDisplaySinkPause *vncWiFiDisplaySinkPause;

  /**
   * \brief Resumes a paused VNC WiFi Display Sink.
   */
  VNCWiFiDisplaySinkResume *vncWiFiDisplaySinkResume;

  /**
   * \brief Informs the SDK that a VNC WiFi Display Sink has lost the input
   * focus.
   */
  VNCWiFiDisplaySinkLostFocus *vncWiFiDisplaySinkLostFocus;

  /**
   * \brief Sends a MirrorLink key event over a VNC WiFi Display Sink's UIBC
   * MirrorLink connection.
   */
  VNCWiFiDisplaySinkSendMirrorLinkKeyEvent *vncWiFiDisplaySinkSendMirrorLinkKeyEvent;

  /**
   * \brief Sends a MirrorLink device key event over a VNC WiFi Display Sink's UIBC
   * MirrorLink connection.
   */
  VNCWiFiDisplaySinkSendMirrorLinkDeviceKeyEvent *vncWiFiDisplaySinkSendMirrorLinkDeviceKeyEvent;

  /**
   * \brief Sends a MirrorLink pointer event over a VNC WiFi Display Sink's
   * UIBC connection.
   */
  VNCWiFiDisplaySinkSendMirrorLinkPointerEvent *vncWiFiDisplaySinkSendMirrorLinkPointerEvent;

  /**
   * \brief Sends a MirrorLink touch event over a VNC WiFi Display Sink's UIBC
   * MirrorLink connection.
   */
  VNCWiFiDisplaySinkSendMirrorLinkTouchEvent *vncWiFiDisplaySinkSendMirrorLinkTouchEvent;

  /**
   * \brief Sends a MirrorLink sink status event over a VNC WiFi Display Sink's
   * UIBC connection.
   */
  VNCWiFiDisplaySinkSendMirrorLinkSinkStatusEvent *vncWiFiDisplaySinkSendMirrorLinkSinkStatusEvent;

  /**
   * \brief Sends a MirrorLink UI blocking event over a VNC WiFi Display Sink's
   * UIBC connection.
   */
  VNCWiFiDisplaySinkSendMirrorLinkUIBlockingEvent *vncWiFiDisplaySinkSendMirrorLinkUIBlockingEvent;

  /**
   * \brief Sends a MirrorLink audio blocking event over a VNC WiFi Display
   * Sink's UIBC connection.
   */
  VNCWiFiDisplaySinkSendMirrorLinkAudioBlockingEvent *vncWiFiDisplaySinkSendMirrorLinkAudioBlockingEvent;

  /**
   * \brief Sends a MirrorLink sink cut text event over a VNC WiFi Display
   * Sink's UIBC connection.
   */
  VNCWiFiDisplaySinkSendMirrorLinkSinkCutTextEvent *vncWiFiDisplaySinkSendMirrorLinkSinkCutTextEvent;

  /**
   * \brief Translates a Unicode character into an X key symbol.
   */
  VNCWiFiDisplaySinkUnicodeToXKeySymbol *vncWiFiDisplaySinkUnicodeToXKeySymbol;

  /**
   * \brief Translates an X key symbol into a Unicode character.
   */
  VNCWiFiDisplaySinkXKeySymbolToUnicode *vncWiFiDisplaySinkXKeySymbolToUnicode;

} VNCWiFiDisplaySDK;

/**
 * \brief The type of VNCWiFiDisplaySDKInitialize().
 *
 * If you are dynamically loading the SDK at runtime, you should use this
 * typedef to declare your function pointer.
 *
 * \see VNCWiFiDisplaySDKInitialize
 */
typedef VNCWiFiDisplayError VNCCALL
VNCWiFiDisplaySDKInitializeType(VNCWiFiDisplaySDK *sdk,
                                size_t sdkSize);

/**
 * \brief The entry point to the SDK.
 *
 * This function is the sole entry point exported by the SDK. Before you can
 * call any other of the SDK's functions, you must call this function to obtain
 * pointers to their implementations.
 *
 * Before unloading the SDK, you should call VNCWiFiDisplaySDKUninitialize().
 *
 * \param sdk Pointer to the VNCWiFiDisplaySDK structure to be initialized.
 * \param sdkSize Size of the VNCWiFiDisplaySDK structure to be initialized.
 *
 * \return VNCWiFiDisplayErrorNone if successful, or an explanatory error
 * code otherwise.
 *
 * \see VNCWiFiDisplaySDK, VNCWiFiDisplaySDKUninitialize,
 * VNCWiFiDisplaySDKInitializeType, VNCVerifyIntegerTypes
 */
VNCDLLIMPORT VNCWiFiDisplayError VNCCALL
VNCWiFiDisplaySDKInitialize(VNCWiFiDisplaySDK *sdk,
                            size_t sdkSize);

#ifdef __cplusplus
}
#endif

#endif  // UUID_a8f7d1c8_0129_11e4_8b37_f7f7140abe98
